/*:
 * @target MZ
 * @plugindesc CGC: Offscreen Dummy Hover (v1.0.3) — zero-frame deselect, UI-safe
 * @author Jordan / Claude
 * @help
 * ■ What this plugin does
 *   - Eliminates the 1–2 frame “flash” where the first card is briefly hovered
 *     at the start of skill selection. Instead, an offscreen/dummy slot is
 *     selected immediately, so no real card is hovered for even a single frame.
 *   - Keeps focus on real UI: if the mouse is over End Turn, Item, Deck,
 *     Discard, Zone sprites, or any extra battle buttons, the plugin does not
 *     steal selection back to the dummy.
 *   - Blocks OK/confirm while the dummy slot is selected (so it can’t be “used”).
 *   - Respects the Item List window: if the Item list is active, the plugin
 *     defers to it and avoids interfering with input.
 *   - Skips card-sprite reordering when the dummy is selected; real cards still
 *     reorder normally for proper z-depth during drag/hover.
 *
 * ■ Installation / Load Order
 *   1) Place this plugin file AFTER:  MYTH_CGC_CoreEngine
 *   2) Keep it AFTER any plugin that changes Window_BattleSkill.maxItems or
 *      Scene_Battle.commandSkill, so our patches run last.
 *   3) No parameters. Works automatically in CGC card battles.
 *
 * ■ How it works (technical)
 *   - Adds one invisible “dummy” slot to the end of the skill list during card
 *     battles (by extending maxItems). That index is treated as “offscreen.”
 *   - On entering Skills (commandSkill), we immediately select the dummy index
 *     in the same tick. This removes the visible card-hover flash entirely.
 *   - We also re-implement Window_BattleSkill.select so that core’s sprite
 *     reordering is performed only for real indices (not the dummy).
 *   - During idle hover updates, if the mouse isn’t over any card or battle UI,
 *     the selection is pushed back to the dummy to avoid lingering hover.
 *
 * ■ Compatibility Notes
 *   - Requires CGC core (MYTH_CGC_CoreEngine). Load after it.
 *   - If another plugin overwrites Scene_Battle.commandSkill or
 *     Window_BattleSkill.select later in the order, our zero-frame fix may be
 *     bypassed. In that case, move this plugin lower in the list.
 *
 * ■ Changelog
 *   v1.0.3  — Respect End Turn & Item list; avoid stealing focus; idle hover tidy.
 */

(() => {
  "use strict";

  //-----------------------------------------------------------------------------
  // Helpers
  //-----------------------------------------------------------------------------
  const hasFn  = (o, k) => !!(o && typeof o[k] === "function");
  const hasArr = (o, k) => !!(o && Array.isArray(o[k]));
  const scn = () => SceneManager._scene;

  // Detect if an Item list window is open & active (so we don't interfere)
  function itemListActive() {
    const sc = scn();
    if (!sc) return false;
    const cands = [
      sc._itemWindow,
      sc._battleItemWindow,
      sc._cardItemWindow,
      sc._cgcItemWindow
    ];
    for (const w of cands) {
      if (w && w.visible && w.active && w.openness === 255) return true;
    }
    return false;
  }

  function getCardSprites(win) {
    const g = win._spritegroupCards || win._cardSprites || scn()?._cardSprites;
    if (!g) return [];
    if (hasFn(g, "getCardSprites")) return g.getCardSprites() || [];
    if (hasArr(g, "_cardSprites"))   return g._cardSprites;
    if (Array.isArray(g))            return g;
    return [];
  }

  function spriteHit(s) {
    if (!s || !s.visible || !s.parent) return false;
    if (hasFn(s, "isTouchedInsideFrame")) return s.isTouchedInsideFrame();
    const b = s.getBounds?.();
    return !!(b && b.contains(TouchInput.x, TouchInput.y));
  }

  function mouseOverAnyCard(win) {
    const list = getCardSprites(win);
    for (let i = list.length - 1; i >= 0; i--) if (spriteHit(list[i])) return true;
    return false;
  }

  function mouseOverAnyBattleUI(win) {
    if (spriteHit(win._endTurnButton)) return true;
    if (spriteHit(win._itemButton))    return true;
    if (spriteHit(win._deckSprite))    return true;
    if (spriteHit(win._discardSprite)) return true;
    const eb = win._extraButtons || [];
    for (let i = eb.length - 1; i >= 0; i--) if (spriteHit(eb[i])) return true;
    const zs = win._zoneSprites || [];
    for (let i = zs.length - 1; i >= 0; i--) if (spriteHit(zs[i])) return true;
    return false;
  }

  //-----------------------------------------------------------------------------
  // Dummy slot plumbing on the skill window
  //-----------------------------------------------------------------------------

  // Base maxItems and computed dummy index
  const _base_maxItems = Window_BattleSkill.prototype.maxItems;
  Window_BattleSkill.prototype.maxItems = function() {
    const base = _base_maxItems.call(this);
    if (!$gameSystem || !$gameSystem._cardBattleEnabled) return base;
    return base + 1; // add one invisible slot at the end
  };

  function dummyIndex(win) { return _base_maxItems.call(win); }

  // Block OK when dummy is selected (unless an Item list is handling input)
  const _processOk = Window_BattleSkill.prototype.processOk;
  Window_BattleSkill.prototype.processOk = function() {
    if (itemListActive()) return; // let Item list handle OK
    if ($gameSystem?._cardBattleEnabled && this.index() === dummyIndex(this)) return;
    _processOk.call(this);
  };

  // On open/activate, snap to dummy immediately
  const _open = Window_BattleSkill.prototype.open;
  Window_BattleSkill.prototype.open = function(...args) {
    const r = _open.apply(this, args);
    if ($gameSystem?._cardBattleEnabled && this.active && !this.previewOnly) {
      const di = dummyIndex(this);
      if (this.index() !== di) this.select(di);
      this._helpWindow?.clear?.();
    }
    return r;
  };

  const _activate = Window_BattleSkill.prototype.activate;
  Window_BattleSkill.prototype.activate = function() {
    _activate.call(this);
    if ($gameSystem?._cardBattleEnabled && !this.previewOnly) {
      const di = dummyIndex(this);
      if (this.index() !== di) this.select(di);
      this._helpWindow?.clear?.();
    }
  };

  // Re-implement select() so we only reorder sprites for real cards
  const _origSelect = Window_BattleSkill.prototype.select;
  Window_BattleSkill.prototype.select = function(index) {
    _origSelect.call(this, index);
    try {
      if ($gameSystem?._cardBattleEnabled && !this.previewOnly) {
        const di = dummyIndex(this);
        if (index !== di && Number.isFinite(this._itemsBeforeCards)) {
          const sc = scn();
          if (sc && typeof sc.reorderCardSprites === "function") {
            sc.reorderCardSprites(index - this._itemsBeforeCards);
          }
        }
      }
    } catch (e) {
      // ignore
    }
  };

  // Idle hover tidy: if not over any card or battle UI, keep dummy selected
  const _update = Window_BattleSkill.prototype.update;
  Window_BattleSkill.prototype.update = function() {
    _update.call(this);
    if (!$gameSystem?._cardBattleEnabled || this.previewOnly) return;
    if (!this.active || itemListActive()) return;
    if (mouseOverAnyBattleUI(this)) return; // respect buttons/extra UI
    if (!mouseOverAnyCard(this) && this.isCursorMovable()) {
      const di = dummyIndex(this);
      if (this.index() !== di) {
        this.select(di);
        this._helpWindow?.clear?.();
      }
    }
  };

  //-----------------------------------------------------------------------------
  // Zero-frame deselect at the exact moment Skill command is entered
  //-----------------------------------------------------------------------------
  const _CGC_CM_commandSkill = Scene_Battle.prototype.commandSkill;
  Scene_Battle.prototype.commandSkill = function() {
    _CGC_CM_commandSkill.call(this); // CGC core may preselect first card here
    try {
      if (!$gameSystem?._cardBattleEnabled) return;
      const w = this._skillWindow;
      if (w && w.active && !w.previewOnly) {
        const di = dummyIndex(w);
        if (w.index() !== di) w.select(di);
        w._helpWindow?.clear?.();
      }
    } catch (e) {
      console.warn("[CGC_ClearMouse v1.0.3] commandSkill patch error:", e);
    }
  };
})();
